% file: example3.m	G. Clifford and G. Moody	27 November 2003

fprintf('example3.m - demonstration of writing annotations using WFDB_tools\n')

% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
% This script is part of a brief 'hands on' tutorial for the WFDB_tools
% package.  Before running this script, you will need to have correctly
% installed the WFDB software package and added the Matlab WFDB_tools
% path at the Matlab prompt.
%
% Please report any problems with this script to Gari Clifford:
%	gari AT physionet DOT org
% Please report any problems with WFDB_tools for Matlab to Jonas Carlson:
%       jonas DOT carlson AT kard DOT lu DOT se
% Please report any problems with the WFDB software package to George Moody:
%       george AT physionet DOT org

% Set pauseflag to 0 before running this script to disable pauses.
if(~exist('pauseflag')) pauseflag=1; end

% Check that the WFDB_tools functions are installed and accessible.
if(exist('WFDB_isigopen')==0)
  fprintf('Please add WFDB_tools to the PATH ...');
  if(isunix)
     error(' e.g.,\n     addpath ~/matlab/WFDB_tools/\n');
  elseif(ispc)
     error(' e.g.,\n     addpath c:\\matlab\\WFDB_tools\n');
  else
     error('... ');
  end
end

% Check that the WFDB software package is installed and accessible.
if system('rdann -r mitdb/201 -a atr -c 1')
  fprintf('The WFDB software package does not seem to be installed.\n');
  error('Install it and add the directory containing "rdann" to your PATH.\n');
end

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

% End of standard setup.
% %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%

fprintf('This demonstration annotates the first two P-waves of record\n');
fprintf('mitdb/201, located at sample numbers 315 and 610 (approximately).\n');

% Look up the anntyp [sic] value corresponding to the "p" mnemonic;  we'll
% need this below.
fprintf('WFDB_strann(%sp%s) gives the annotation code corresponding to "p".\n',setstr(39),setstr(39));
fprintf('We%sll save this value as PWAVE for later use.\n',setstr(39));
PWAVE = WFDB_strann('p');
fprintf('WFDB_strann(%sp%s): ',setstr(39),setstr(39));
PWAVE

fprintf('WFDB_anndesc(PWAVE) gives a description of what this value means\n');

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('WFDB_anndesc(PWAVE): ');
WFDB_anndesc(PWAVE) % P-wave peak !

% Prepare to create an empty annotation file in which to write the output.
fprintf('Create an annotator information structure using ...\n');
fprintf('  A = WFDB_Anninfo(1)\n');
A = WFDB_Anninfo(1);

fprintf('We%sll use the annotator name "p", so we fill the structure by:\n',setstr(39));
fprintf('  A.name = %sp%s; \n',setstr(39),setstr(39)); 
fprintf('  A.stat = %sWFDB_WRITE%s; \n \n',setstr(39),setstr(39)); 
A.name = 'p';
A.stat = 'WFDB_WRITE';

fprintf('Create an empty annotation file using ...\n');
fprintf('  WFDB_annopen(%smitdb/201%s,A)\n',setstr(39),setstr(39));
WFDB_annopen('mitdb/201',A);

fprintf('Next, we create two generic annotations in memory using: \n')
fprintf('  ann = WFDB_Annotation(2)\n');
ann = WFDB_Annotation(2)

fprintf('\nThe contents of ann look like ...\n')
ann

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

% Copy the data into the annotations.
fprintf('\nNow we explicitly set the elements of the annotation ...\n')

fprintf('ann(1).time = 315; \n'); 
fprintf('ann(1).anntyp = PWAVE;  \n'); 
fprintf('ann(1).aux = %sFirst P-wave%s;  \n',setstr(39),setstr(39)); 
fprintf('ann(2).time = 610;  \n');
fprintf('ann(2).anntyp = PWAVE;  \n');
fprintf('ann(2).aux = %sSecond P-wave%s;   \n \n',setstr(39),setstr(39));

ann(1).time = 315;   
ann(1).anntyp = PWAVE;
ann(1).aux = 'First P-wave';
ann(2).time = 610;
ann(2).anntyp = PWAVE;
ann(2).aux = 'Second P-wave'; 

fprintf('\nThe contents of ann are now:\n')
ann

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('Now write the annotations to annotator 0 (mitdb/201.p) using ...\n');
fprintf('   WFDB_putann(0,ann);  \n')

WFDB_putann(0,ann);

if(pauseflag==1)
  fprintf('               [Press any key to continue]\n'); 
  pause;
end;

fprintf('We finish by closing all files with WFDB_wfdbquit\n');
fprintf('This creates a file mitdb/201.p that WFDB applications can read\n')
WFDB_wfdbquit              % close all open files 

if(pauseflag==1)
  fprintf('Press any key to view the results using WAVE or GTKWave\n');
  fprintf('if either is available: ');
  pause;
end;

if system('wave -r mitdb/201 -a p -c 1')
  if system('gtkwave -r mitdb/201 -a p -c 1')
    fprintf('Neither WAVE nor GTKWave appears to be installed.  Here is\n');
    fprintf('a text version of the results produced using "rdann":\n\n');
    if system('rdann -r mitdb/201 -a p')
      fprintf('rdann was not successful\n');
    end;
  end;
end;

fprintf('End of example3.m\n');
