#include "mex.h"
#include "matrix.h"
#include "wfdb/wfdb.h"

/* Define order of input variables */
#define RECORD_INPUT 0
#define NSIG_INPUT 1

/* Define order of output variables */
#define STRUCT_OUTPUT 0

void mexFunction( int nlhs, mxArray *plhs[],
                  int nrhs, const mxArray *prhs[] )
{
  const char **fnames;
  int nsig, err_code;
  char *record;
  unsigned int record_length, i;
  WFDB_Siginfo *siarray;
  mxArray *value;

  /* Check for correct total number of input arguments. */
  if (nrhs != 2)
    mexErrMsgTxt("Two input arguments required.");
      
  /* Create output structure with same fields as a WFDB_Siginfo */
  fnames = mxCalloc(14, sizeof(*fnames));
  fnames[0]  = "fname";
  fnames[1]  = "desc";
  fnames[2]  = "units";
  fnames[3]  = "gain";
  fnames[4]  = "initval";
  fnames[5]  = "group";
  fnames[6]  = "fmt";
  fnames[7]  = "spf";
  fnames[8]  = "bsize";
  fnames[9]  = "adcres";
  fnames[10] = "adczero";
  fnames[11] = "baseline";
  fnames[12] = "nsamp";
  fnames[13] = "cksum";

  /* RECORD input must be a string. */
  if (!mxIsChar(prhs[RECORD_INPUT]))
    mexErrMsgTxt("RECORD must be a string.");

  /* Get the RECORD string. */
  record_length = (mxGetM(prhs[RECORD_INPUT]) *
		     mxGetN(prhs[RECORD_INPUT])) + 1;
  record = mxCalloc(record_length, sizeof(char));
  mxGetString(prhs[RECORD_INPUT], record, record_length);

  /* Get number of signals */
  nsig = *mxGetPr(prhs[NSIG_INPUT]);

  siarray = (WFDB_Siginfo *)mxMalloc(nsig * sizeof(WFDB_Siginfo));

  /* Open output file */
  if ( (err_code = osigopen(record, siarray, nsig)) < 0)
    switch (err_code)
      {
      case -1:
	mexErrMsgTxt("Failure: unable to read header file");
      case -2:
	mexErrMsgTxt("Failure: incorrect header file format");
      case -3:
	mexErrMsgTxt("Failure: unable to open output signal(s)"); 
      default:
	mexPrintf("Error code: %d\n", nsig);
	mexErrMsgTxt("Unknown error");
      }

  if (err_code != nsig)
    mexErrMsgTxt("Number of outputs does not match NSIG");

  /* Create output structure */
  plhs[STRUCT_OUTPUT] = mxCreateStructMatrix(nsig, 1, 14, fnames);

  /* Fill output structures */
  for (i = 0; i < nsig; i++)
    {
      mxSetField(plhs[0], i, "fname", mxCreateString(siarray[i].fname));

      mxSetField(plhs[0], i, "desc", mxCreateString(siarray[i].desc));

      mxSetField(plhs[0], i, "units", mxCreateString(siarray[i].units));

      mxSetField(plhs[0], i, "gain", 
		 mxCreateDoubleScalar(siarray[i].gain));

      mxSetField(plhs[0], i, "initval", 
		 mxCreateDoubleScalar(siarray[i].initval));

      mxSetField(plhs[0], i, "group", 
		 mxCreateDoubleScalar(siarray[i].group));

      mxSetField(plhs[0], i, "fmt", 
		 mxCreateDoubleScalar(siarray[i].fmt));

      mxSetField(plhs[0], i, "spf", 
		 mxCreateDoubleScalar(siarray[i].spf));

      mxSetField(plhs[0], i, "bsize", 
		 mxCreateDoubleScalar(siarray[i].bsize));

      mxSetField(plhs[0], i, "adcres", 
		 mxCreateDoubleScalar(siarray[i].adcres));

      mxSetField(plhs[0], i, "adczero", 
		 mxCreateDoubleScalar(siarray[i].adczero));

      mxSetField(plhs[0], i, "baseline", 
		 mxCreateDoubleScalar(siarray[i].baseline));

      mxSetField(plhs[0], i, "nsamp", 
		 mxCreateDoubleScalar(siarray[i].nsamp));

      mxSetField(plhs[0], i, "cksum", 
		 mxCreateDoubleScalar(siarray[i].cksum));

    }
}
